﻿namespace Hims.Api.Controllers
{
    using System;
    using System.Collections.Generic;
    using System.Threading.Tasks;
    using Domain.Services;
    using Hims.Api.Models;
    using Hims.Shared.UserModels.Common;
    using Hims.Shared.UserModels.Slots;
    using Microsoft.AspNetCore.Authorization;
    using Microsoft.AspNetCore.Mvc;
    using Shared.EntityModels;
    using Shared.UserModels.NurseNote;
    using Utilities;

    /// <inheritdoc />
    [Authorize]
    [Route("api/nurse-note")]
    [Consumes("application/json")]
    [Produces("application/json")]
    public class NurseNoteController : BaseController
    {
        /// <summary>
        /// The coupon services.
        /// </summary>
        private readonly INurseNoteService service;

        /// <inheritdoc />
        public NurseNoteController(INurseNoteService service)
        {
            this.service = service;
        }

        /// <summary>
        /// The fetch coupons.
        /// </summary>
        /// <param name="model">
        /// The coupon filter model.
        /// </param>
        /// <returns>
        /// The list of coupons.
        /// </returns>
        /// <remarks>
        /// ### REMARKS ###
        /// The following codes are returned
        /// - 200 - List of coupons.
        /// - 500 - Problem with Server side code.
        /// </remarks>
        [HttpPost]
        [Authorize]
        [Route("fetch")]
        [ProducesResponseType(typeof(List<BedModel>), 200)]
        [ProducesResponseType(500)]
        public async Task<ActionResult> FetchAsync([FromBody] FetchPayload model, [FromHeader] LocationHeader header)
        {
            var data = await this.service.FetchAsync(model);
            return this.Success(new GenericResponse()
            {
                Status = GenericStatus.Success,
                Data = data
            });
        }


        /// <summary>
        /// The add coupon.
        /// </summary>
        /// <param name="model">
        /// The model.
        /// </param>
        /// <returns>
        /// The <see cref="Task"/>.
        /// </returns>
        /// <remarks>
        /// ### REMARKS ###
        /// The following codes are returned
        /// - 200 - Coupon added successfully.
        /// - 409 - Coupon already exist.
        /// - 500 - Problem with Server side code.
        /// </remarks>
        [HttpPost]
        [Route("insert")]
        [ProducesResponseType(typeof(string), 200)]
        [ProducesResponseType(409)]
        [ProducesResponseType(500)]
        public async Task<ActionResult> AddAsync([FromBody] InsertModel model, [FromHeader] LocationHeader header)
        {
            var response = await this.service.InsertAsync(model);
            return this.Success(new GenericResponse()
            {
                Status = response > 0 ? GenericStatus.Success : GenericStatus.Error
            });
        }
    }
}